/**
 * A client-side 802.1x implementation 
 *
 * This code is released under both the GPL version 2 and BSD licenses.
 * Either license may be used.  The respective licenses are found below.
 *
 * Copyright (C) 2002 Bryan D. Payne & Nick L. Petroni Jr.
 * All Rights Reserved
 *
 * --- GPL Version 2 License ---
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * --- BSD License ---
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the University of
 *       Maryland at College Park and its contributors.
 *  - Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*******************************************************************
 * The driver function for a Linux application layer EAPOL 
 * implementation
 * File: profile.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: profile.c,v 1.21 2004/08/10 01:59:26 chessing Exp $
 * $Date: 2004/08/10 01:59:26 $
 * $Log: profile.c,v $
 * Revision 1.21  2004/08/10 01:59:26  chessing
 *
 * Added support for the SNMP supplicant counters defined in the IEEE 802.1X-2001 document.
 *
 * Revision 1.20  2004/08/05 23:56:17  chessing
 *
 * Added basic support for scanning for broadcast SSIDs.  This is another step closer to WPA/11i support. ;)
 *
 * Revision 1.19  2004/07/25 19:36:40  chessing
 * Fixed a few more logical/bitwise nots.  Added rtnetlink support for hot-plugging interfaces.  (Hot unplugging is coming soon. ;)
 *
 * Revision 1.18  2004/07/19 02:43:16  chessing
 *
 * Changed things to get rid of Linux specific pieces in the interface_data struct. Fixed up EAP-SIM and EAP-AKA to skip AVPs that we don't support.  (We print a mesage, and move on.)  Added the --enable-radiator-test option to configure EAP-AKA to use the test vectors included with Radiator's AKA module.  (To use this mode, no SIM card is required, but the PCSC library is still needed to build.  Also, some errors will be displayed.)
 *
 * Revision 1.17  2004/07/15 04:15:35  chessing
 *
 * True/false int values are now bit flags in a byte.  PEAP now calls back in to functions from eap.c for phase 2 methods.  This allows for any phase 1 EAP type to work.  This resulted in some major changes the functions in eap.c, and in peap_phase2.c.  PEAP has been tested using both MS-CHAPv2, and EAP-MD5 as inner types.  More types need to be tested, and enabled.
 *
 * Revision 1.16  2004/06/15 03:22:29  chessing
 *
 * XSupplicant Release 1.0
 *
 *
 *******************************************************************/

#include <stdlib.h>
#include <strings.h>
#include <string.h>

#include "snmp.h"
#include "profile.h"
#include "xsup_err.h"
#include "cardif/cardif.h"
#include "xsup_debug.h"
#include "eapol.h"

/*******************************************
 *
 * Initalize the default values for the structure.  In general, state machine
 * and user configuration variables won't need to be set here.  We should
 * set up variables that are in the root of the structure.
 *
 *******************************************/
int init_interface_struct(struct interface_data *work, char *intname)
{
  char dot1x_default_dest[6] = {0x01, 0x80, 0xc2, 0x00, 0x00, 0x03};

  if (!work)
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed in to init_interface_struct()!\n");
      return XEMALLOC;
    }

  if (!intname)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface name passed in to init_interface_struct()!\n");
      return XEMALLOC;
    }

  work->intName = (char *)malloc(strlen(intname)+1);
  if (work->intName == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory in %s at line %d!\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }
  bzero(work->intName, strlen(intname)+1);
  strncpy(work->intName, intname, strlen(intname));

  debug_printf(DEBUG_INT, "ADDING INTERFACE : %s\n", intname);

  bzero(work->source_mac, 6);
  bzero(work->dest_mac, 6);

  work->ssid_list = NULL;
  work->keyingMaterial = NULL;

  // Allocate memory for SNMP stats, and set them to 0s.
  work->snmp = (struct snmp_data *)malloc(sizeof(struct snmp_data));
  if (work->snmp == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory in %s at line %d!\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }
  bzero(work->snmp, sizeof(struct snmp_data));

  work->flags = (0x00 | WAS_DOWN | FIRSTAUTH);
  debug_printf(DEBUG_INT, "Flags are : %02X\n", work->flags);

  // The default MAC specified by the IEEE 802.1x spec.
  memcpy(&work->dest_mac[0], &dot1x_default_dest, 6);
	
  work->cur_essid = NULL;

  eapol_init(work);

  work->userdata = NULL;

  work->tempPassword = NULL;

  return XENONE;
}

/**********************************************
 *
 * Clean out a structure.  Clear out everything but return a pointer to "next"
 * the pointer to next should be used in a successive call.
 *
 **********************************************/
struct interface_data *destroy_interface_struct(struct interface_data *intdata)
{
  struct interface_data *next;

  if (!intdata) return NULL;

  // We should *always* have something in intName.
  free(intdata->intName);

  // Ditto.
  free(intdata->snmp);

  if (intdata->cur_essid) free(intdata->cur_essid);
  if (intdata->keyingMaterial) free(intdata->keyingMaterial);

  next = intdata->next;
  return next;
}
