/**
 * A client-side 802.1x implementation supporting EAP/TLS
 *
 * This code is released under both the GPL version 2 and BSD licenses.
 * Either license may be used.  The respective licenses are found below.
 * 
 * Copyright (C) 2002 Bryan D. Payne & Nick L. Petroni Jr.
 * All Rights Reserved
 *
 * --- GPL Version 2 License ---
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * --- BSD License ---
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the University of
 *       Maryland at College Park and its contributors.
 *  - Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*******************************************************************
 * EAPTTLS Function implementations
 * 
 * File: eapttls.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: eapttls.c,v 1.27 2004/06/15 03:35:21 chessing Exp $
 * $Date: 2004/06/15 03:35:21 $
 * $Log: eapttls.c,v $
 * Revision 1.27  2004/06/15 03:35:21  chessing
 *
 * New updates including fixes to LEAP (keying now works with wireless) and adding EAP-AKA.
 *
 * Revision 1.26  2004/06/15 03:22:32  chessing
 *
 * XSupplicant Release 1.0
 *
 *
 *******************************************************************/

#include <string.h>
#include "profile.h"
#include "config.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "frame_structs.h"
#include "eap_types/tls/eaptls.h"
#include "eap_types/ttls/eapttls.h"
#include "eap_types/ttls/ttlsphase2.h"
#include "eap_types/tls/tls_funcs.h"
#include "eap.h"
#include "interactive.h"

int eapttls_setup(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;
  struct config_eap_ttls *userdata;

  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface structure passed to eapttls_setup()!\n");
      return XEMALLOC;
    }

  userdata = (struct config_eap_ttls *)thisint->eap_conf_data;

  if (userdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "EAP-TTLS setup was passed NULL userdata!  We cannot continue with this attempt at authentication!\n");
      return XENOUSERDATA;
    }

  // First, set up the structure to hold all of our instance specific
  // variables.
  thisint->eap_data = (char *)malloc(sizeof(struct tls_vars));
  if (thisint->eap_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Unable to allocate memory for thisint->eapdata in eapttls_setup()!\n");
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;

  // Set our variables to NULL.
  mytls_vars->ctx = NULL;
  mytls_vars->ssl = NULL;
  mytls_vars->ssl_in = NULL;
  mytls_vars->ssl_out = NULL;
  mytls_vars->tlsoutdata = NULL;
  mytls_vars->tlsoutsize = 0;
  mytls_vars->tlsoutptr = 0;
  mytls_vars->cncheck = userdata->cncheck;
  mytls_vars->cnexact = userdata->cnexact;
  mytls_vars->phase = 1;
  mytls_vars->resume = userdata->session_resume;
  mytls_vars->quickResponse = TRUE;
  mytls_vars->cert_loaded = FALSE;
  mytls_vars->verify_mode = SSL_VERIFY_PEER;

  mytls_vars->sessionkeyconst = (char *)malloc(TTLS_SESSION_KEY_CONST_SIZE+1);
  if (mytls_vars->sessionkeyconst == NULL) return XEMALLOC;

  bzero(mytls_vars->sessionkeyconst, TTLS_SESSION_KEY_CONST_SIZE+1);
  strncpy(mytls_vars->sessionkeyconst, TTLS_SESSION_KEY_CONST,
	  TTLS_SESSION_KEY_CONST_SIZE);

  mytls_vars->sessionkeylen = TTLS_SESSION_KEY_CONST_SIZE;

  debug_printf(DEBUG_EVERYTHING, "(EAP-TTLS) Initialized.\n");
  
  if (tls_funcs_init(thisint) != XENONE) 
    {
      debug_printf(DEBUG_NORMAL, "Couldn't initialize OpenSSL!\n");
      return XETLSINIT;
    }

  if (userdata->root_cert == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "userdata->root_cert is NULL!  We cannot continue without a valid certificate!\n");
      return XENOUSERDATA;
    }

  debug_printf(DEBUG_AUTHTYPES, "userdata->root_cert = %s\n", userdata->root_cert);
  if (strcmp(userdata->root_cert, "NONE") == 0)
    {
      // We were told not to verify certificates.  Spew out a warning, and
      // then do it!
      mytls_vars->verify_mode = SSL_VERIFY_NONE;
      debug_printf(DEBUG_NORMAL, "****WARNING**** Turning off certificate verification is a *VERY* bad idea!  You should not use this mode outside of basic testing, as it will compromise the security of your connection!\n");
    } else {
      if (tls_funcs_load_root_certs(thisint, userdata->root_cert, 
				    userdata->root_dir, userdata->crl_dir) != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't load root certificates!\n");
	  return XETLSINIT;
	}
    }

  if ((userdata->user_cert != NULL) && ((userdata->user_key_pass != NULL) ||
				   (thisint->tempPwd != NULL)))
    {
      debug_printf(DEBUG_NORMAL, "Using user certificate with TTLS!\n");
      tls_funcs_load_user_cert(thisint, userdata->user_cert, 
			       userdata->user_key, userdata->user_key_pass,
			       userdata->random_file);

      mytls_vars->cert_loaded = TRUE;
    }

  // Otherwise, if we don't have a user cert, then just set cert_loaded to true!
  if (userdata->user_cert == NULL) mytls_vars->cert_loaded = TRUE;

  return XENONE;
}

int eapttls_process(struct generic_eap_data *thisint, u_char *dataoffs, 
		    int insize, u_char *outframe, int *outsize)
{
  struct config_eap_ttls *userdata;
  struct tls_vars *mytls_vars;
  int result = XENONE;
  int retVal;

  if ((!thisint) || (!thisint->eap_conf_data) || (!thisint->eap_data) ||
      (!outframe))
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed to eapttls_process()!\n");
      return XEMALLOC;
    }

  userdata = (struct config_eap_ttls *)thisint->eap_conf_data;
  mytls_vars = (struct tls_vars *)thisint->eap_data;

  // The state machine wants to know if we have anything else to say.
  // We may be waiting for the server to send us more information, or
  // we may need to send a request to the GUI for a password, and wait
  // for an answer.

  // TTLS is slightly different than others.  Since we don't *need* to have
  // a client certificate to make things work correctly, we may not need
  // a password here.
  if (mytls_vars->cert_loaded == FALSE)
    {
      if (userdata->user_cert != NULL)
	{
	  if ((thisint->tempPwd == NULL) && (userdata->user_key_pass == NULL))
	    {
	      // We need to indicate to the caller that we need a password.
	      thisint->need_password = 1;
	      thisint->eaptype = strdup("EAP-TTLS User Certificate");
	      thisint->eapchallenge = NULL;   // Probably not needed, but just
                                              // to be safe.

	      *outsize = 0;

	      return XENONE;
	    }
	}

      if ((mytls_vars->cert_loaded == FALSE) && ((thisint->tempPwd != NULL) ||
						 (userdata->user_key_pass != NULL)))
      {
	// Load the user certificate.
	if ((retVal = tls_funcs_load_user_cert(thisint, userdata->user_cert, 
					       userdata->user_key,
					       userdata->user_key_pass,
					       userdata->random_file))!=XENONE)
	  {
	    debug_printf(DEBUG_NORMAL, "Error loading user certificate!\n");
	    return retVal;
	  } else {

	    // Otherwise, the certificate is loaded.
	    mytls_vars->cert_loaded = TRUE;

	    // Check if we used tempPwd, if we did, free it, since we don't
	    // need it anymore.
	    if (thisint->tempPwd != NULL)
	      {
		free(thisint->tempPwd);
		thisint->tempPwd = NULL;
	      }
	  }
      }  
    }


  // Make sure we have a valid packet to process.
  if (dataoffs == NULL) return XENONE;
  
  result=tls_funcs_decode_packet(thisint, dataoffs, insize,  outframe, outsize,
				 (phase2_call)ttls_do_phase2, userdata->chunk_size);

  if (result != 0)
    {
      return result;
    }
  return *outsize;
}

int eapttls_get_keys(struct interface_data *thisint)
{
  if (!thisint)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface structure passed to eapttls_get_keys()!\n");
      return XEMALLOC;
    }

  if (thisint->keyingMaterial != NULL)
    {
      free(thisint->keyingMaterial);
    }
  
  thisint->keyingMaterial = tls_funcs_gen_keyblock(thisint->userdata->activemethod);
  thisint->keyingLength = 32;

  if (thisint->keyingMaterial == NULL) return -1;

  return 0;
}

int eapttls_cleanup(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;

  if ((!thisint) || (!thisint->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid structure in eapttls_cleanup()!\n");
      return XEMALLOC;
    }

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  tls_funcs_cleanup(thisint);

  if (mytls_vars != NULL)
    {
      free(mytls_vars);
      mytls_vars = NULL;
    }

  debug_printf(DEBUG_EVERYTHING, "(EAP-TTLS) Cleaned up.\n");
  return XENONE;
}

int eapttls_failed(struct generic_eap_data *thisint)
{
  struct tls_vars *mytls_vars;

  if ((!thisint) || (!thisint->eap_data))
    {
      debug_printf(DEBUG_NORMAL, "Invalid structure passed to eapttls_failed()!\n");
      return XEMALLOC;
    }

  // Call ttls_phase2_failed first.
  ttls_phase2_failed(thisint);

  mytls_vars = (struct tls_vars *)thisint->eap_data;
  tls_funcs_failed(thisint);

  debug_printf(DEBUG_EVERYTHING, "(EAP-TTLS) Failed. Resetting.\n");
  return XENONE;
}
